{ *********************************************************************** }
{                                                                         }
{ Delphi Visual Component Library                                         }
{                                                                         }
{ Copyright (c) 2001-2005 Borland Software Corporation                    }
{                                                                         }
{ *********************************************************************** }

unit Borland.Vcl.WinUtils platform;

interface

uses
  SysUtils, Windows, Types, {$IFNDEF CF}ActiveX,{$ENDIF}Messages;

{ NativeBufToArray copies an array of valuetyes from an unmanaged
  memory buffer to a managed array }

function NativeBufToArray(Buffer: IntPtr; Data: System.Array): System.Array; overload;
function NativeBufToArray(Buffer: IntPtr; Data: System.Array;
  ElementCount: Integer): System.Array; overload;

{ ArrayToNativeBuf copies an array of valuetypes to an unmanaged
  memory buffer. The buffer must be deallocated using
  Marshal.FreeHGlobal }

function ArrayToNativeBuf(Data: System.Array; Buffer: IntPtr = nil): IntPtr;

type
  TWndMethod = procedure(var Message: TMessage) of object;

function MakeObjectInstance(Method: TWndMethod): TFNWndProc;
procedure FreeObjectInstance(ObjectInstance: TFNWndProc);

function AllocateHWnd(Method: TWndMethod): HWND;
procedure DeallocateHWnd(Wnd: HWND);

function PointToLParam(P: TPoint): LPARAM;

function GetCmdShow: Integer;

procedure SafeArrayCheck(AResult: HRESULT);

{ BytesToStructure copies a structure stored in an array of bytes
  into a structure }

function BytesToStructure(const Bytes: TBytes; AType: System.Type): TObject;

{ StructureToBytes copies a structure into an array of bytes }

function StructureToBytes(const Struct: TObject): TBytes;

{ HInstance returns the handle of the module/instance }

function HInstance: HINST;

{ MainInstance returns the handle of the main(.EXE) HInstance }

function MainInstance: HINST;

{ GetNativeVariantsForObjects copies an array of objects to an
  array of native variants. The resulting array must be freed
  using FreeNativeVariants }

{$IFNDEF CF}
function GetNativeVariantsForObjects(const Objects: array of TObject): IntPtr;

{ FinalizeNativeVariants iterates through an array of native
  variants calling VariantClear on each element }

procedure FinalizeNativeVariants(Buffer: IntPtr; Offset, Count: Integer);

{ FreeNativeVariants finalizes each element in an array of
  native variants and free's the unmanaged memory allocated
  to hold it }

procedure FreeNativeVariants(NativeVariants: IntPtr; Count: Integer);
{$ENDIF}

implementation

uses
  SysConst, System.Text, System.Collections, System.Runtime.InteropServices,
{$IFNDEF CF}
  System.Security.Permissions,
{$ELSE}
  CFUtils,
{$ENDIF}
  System.Reflection;

function NativeBufToArray(Buffer: IntPtr; Data: System.Array): System.Array;
begin
  Result := NativeBufToArray(Buffer, Data, Data.Length);
end;

function NativeBufToArray(Buffer: IntPtr; Data: System.Array;
  ElementCount: Integer): System.Array;
var
  I, LowerBound: Integer;
  Size: Longint;
begin
  Result := Data;
  LowerBound := Data.GetLowerBound(0);
  Size := Marshal.SizeOf(Data.GetType.GetElementType);

  for I := 0 to ElementCount - 1 do
  begin
    Result.SetValue(Marshal.PtrToStructure(IntPtr(Longint(Buffer.ToInt32 + (I * Size))),
      Data.GetType.GetElementType), I + LowerBound);
  end;
end;

function ArrayToNativeBuf(Data: System.Array; Buffer: IntPtr = nil): IntPtr;
var
  I, LowerBound: Integer;
  Size: Longint;
begin
  LowerBound := Data.GetLowerBound(0);
  Size := Marshal.SizeOf(Data.GetType.GetElementType);
  with Marshal do
  begin
    if Buffer = nil then
      Result := AllocHGlobal(Size * Data.Length)
    else
      Result := Buffer;
    try
      for I := 0 to Data.Length - 1 do
        StructureToPtr(TObject(Data.GetValue(I + LowerBound)),
          IntPtr(Longint(Result.ToInt32 + (I * Size))), False);
    except
      if Buffer = nil then FreeHGlobal(Result);
      raise;
    end;
  end;
end;

const
  InstanceCount = 313;

{ Object instance management }

type
  TObjectInstance = class
    FWndMethod: TWndMethod;
    function WndProc(Handle: HWND; Msg: UINT; WParam: WPARAM; LParam: LPARAM): LRESULT;
    constructor Create(WndMethod: TWndMethod);
  end;

constructor TObjectInstance.Create(WndMethod: TWndMethod);
begin
  inherited Create;
  FWndMethod := WndMethod;
end;

function TObjectInstance.WndProc(Handle: HWND; Msg: UINT; WParam: WPARAM; LParam: LPARAM): LRESULT;
var
  Message: TMessage;
begin
  Message := TMessage.Create(Msg, WParam, LParam);
  FWndMethod(Message);
  Result := Message.Result;
end;

function MakeObjectInstance(Method: TWndMethod): TFNWndProc;
var
  Instance: TObjectInstance;
begin
  Instance := TObjectInstance.Create(Method);
  Result := Instance.WndProc;
end;

procedure FreeObjectInstance(ObjectInstance: TFNWndProc);
begin
 // Nothing to do
end;

var
  UtilWindowClass: TWndClass = (
    style: 0;
    lpfnWndProc: @DefWindowProc;
    cbClsExtra: 0;
    cbWndExtra: 0;
    hInstance: 0;
    hIcon: 0;
    hCursor: 0;
    hbrBackground: 0;
    lpszMenuName: '';
    lpszClassName: 'TPUtilWindow');
  Instances: Hashtable;

{$IFNDEF CF}
[SecurityPermission(SecurityAction.LinkDemand, UnmanagedCode=True)]
{$ENDIF}
function AllocateHWnd(Method: TWndMethod): HWND;
var
  TempClassInfo: TWndClassInfo;
  ClassRegistered: Boolean;
  Instance: TFNWndProc;
begin
  UtilWindowClass.hInstance := HInstance;
  UtilWindowClass.lpszClassName := Format('%s.%d',
{$IFNDEF CF}
    [UtilWindowClass.lpszClassName, AppDomain.CurrentDomain.GetHashCode]);
{$ELSE}
    [UtilWindowClass.lpszClassName, 0]);
{$ENDIF}
  ClassRegistered := GetClassInfo(HInstance, UtilWindowClass.lpszClassName,
    TempClassInfo);
  if not ClassRegistered {or (TempClass.lpfnWndProc <> @DefWindowProc)} then
    RegisterClass(UtilWindowClass);
  Result := CreateWindowEx(WS_EX_TOOLWINDOW, UtilWindowClass.lpszClassName,
    '', WS_POPUP {+ 0}, 0, 0, 0, 0, 0, 0, HInstance, nil);
  if Assigned(Method) then
  begin
    Instance := MakeObjectInstance(Method);
    if not Assigned(Instances) then
      Instances := Hashtable.Create;
    Instances[TObject(Result)] := @Instance;
    SetWindowLong(Result, GWL_WNDPROC, @Instance);
  end;
end;

{$IFNDEF CF}
[SecurityPermission(SecurityAction.LinkDemand, UnmanagedCode=True)]
{$ENDIF}
procedure DeallocateHWnd(Wnd: HWND);
begin
  DestroyWindow(Wnd);
  if Instances <> nil then
    Instances[TObject(Wnd)] := nil;
end;

function PointToLParam(P: TPoint): LPARAM;
begin
  Result := LongInt((P.X and $0000ffff) or (P.Y shl 16))
end;

function GetCmdShow: Integer;
{$IFNDEF CF}
var
  SI: TStartupInfo;
{$ENDIF}
begin
  Result := 10;                  { SW_SHOWDEFAULT }
{$IFNDEF CF}
  GetStartupInfo(SI);
  if SI.dwFlags and 1 <> 0 then  { STARTF_USESHOWWINDOW }
    Result := SI.wShowWindow;
{$ENDIF}
end;

{ SafeArray Utility Functions }

type
  ESafeArrayError = class(Exception)
  private
    FErrorCode: HRESULT;
  public
    constructor CreateHResult(AResult: HRESULT; const AMessage: string = '');
    property ErrorCode: HRESULT read FErrorCode write FErrorCode;
  end;

  ESafeArrayBoundsError = class(ESafeArrayError);
  ESafeArrayLockedError = class(ESafeArrayError);

const
  VAR_BADINDEX      = HRESULT($8002000B); // = Windows.DISP_E_BADINDEX
  VAR_ARRAYISLOCKED = HRESULT($8002000D); // = Windows.DISP_E_ARRAYISLOCKED

constructor ESafeArrayError.CreateHResult(AResult: HRESULT; const AMessage: string);
var
  S: string;
begin
  S := AMessage;
  if S = '' then
    S := Format(SVarArrayWithHResult, [AResult]);
  Create(S);
  FErrorCode := AResult;
end;

procedure SafeArrayError(AResult: HRESULT);
begin
  case AResult of
    VAR_BADINDEX:      raise ESafeArrayBoundsError.CreateHResult(AResult, SVarArrayBounds);
    VAR_ARRAYISLOCKED: raise ESafeArrayLockedError.CreateHResult(AResult, SVarArrayLocked);
  else
    raise ESafeArrayError.CreateHResult(AResult);
  end;
end;

procedure SafeArrayCheck(AResult: HRESULT);
begin
  if AResult and $80000000 <> 0 then
    SafeArrayError(AResult);
end;

function BytesToStructure(const Bytes: TBytes; AType: System.Type): TObject;
var
  Size: Integer;
  Buffer: IntPtr;
begin
  Size := Marshal.SizeOf(AType);
  Buffer := Marshal.AllocHGlobal(Size);
  try
    Marshal.Copy(Bytes, 0, Buffer, Size);
    Result := Marshal.PtrToStructure(Buffer, AType);
  finally
    Marshal.FreeHGlobal(Buffer);
  end;
end;

function StructureToBytes(const Struct: TObject): TBytes;
var
  Buffer: IntPtr;
begin
  Buffer := Marshal.AllocHGlobal(Marshal.SizeOf(TypeOf(Struct)));
  try
    Marshal.StructureToPtr(Struct, Buffer, False);
    SetLength(Result, Marshal.SizeOf(TypeOf(Struct)));
    Marshal.Copy(Buffer, Result, 0, Length(Result));
  finally
    Marshal.FreeHGlobal(Buffer);
  end;
end;

                       
function HInstance: HINST;
begin
{$IFNDEF CF}
  Result := HINST(Marshal.GetHInstance(Assembly.GetCallingAssembly.GetModules[0]));
{$ELSE}
  Result :=0;
{$ENDIF}
end;

                           
function MainInstance: HINST;
begin
{$IFNDEF CF}
  if not IsLibrary then
    Result := HINST(Marshal.GetHInstance(Assembly.GetEntryAssembly.GetModules[0]))
  else
{$ENDIF}
    Result := 0;
end;

const
  NativeVariantSize = 16;

{$IFNDEF CF}
function GetNativeVariantsForObjects(const Objects: array of TObject): IntPtr;
var
  I: Integer;
begin
  Result := nil;
  if not Assigned(Objects) then Exit;
  Result := Marshal.AllocHGlobal(Length(Objects) * NativeVariantSize);
  for I := 0 to Length(Objects) - 1 do
    Marshal.GetNativeVariantForObject(Objects[I],
      IntPtr.Create(Result.ToInt32 + (I * NativeVariantSize)));
end;

procedure FinalizeNativeVariants(Buffer: IntPtr; Offset, Count: Integer);
var
  I: Integer;
begin
  for I := 0 to Count - 1 do
    VariantClear(IntPtr.Create(Buffer.ToInt32 + Offset + (I * NativeVariantSize)));
end;

procedure FreeNativeVariants(NativeVariants: IntPtr; Count: Integer);
begin
  FinalizeNativeVariants(NativeVariants, 0, Count);
  Marshal.FreeHGlobal(NativeVariants);
end;
{$ENDIF}

end.
